/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: SCF CardEvent sample.
 *
 */

package scfevent;

import jp.co.ricoh.dsdk.panel.AttributeType;
import jp.co.ricoh.dsdk.panel.Container;
import jp.co.ricoh.dsdk.panel.Font;
import jp.co.ricoh.dsdk.panel.Frame;
import jp.co.ricoh.dsdk.panel.LED;
import jp.co.ricoh.dsdk.panel.LEDExecutionException;
import jp.co.ricoh.dsdk.panel.Label;
import jp.co.ricoh.dsdk.panel.Text;
import jp.co.ricoh.dsdk.panel.Window;
import jp.co.ricoh.dsdk.panel.LED.Operation;
import jp.co.ricoh.dsdk.panel.LED.Type;
import jp.co.ricoh.dsdk.panel.event.KeyAdapter;
import jp.co.ricoh.dsdk.panel.event.KeyEvent;
import jp.co.ricoh.dsdk.scard.framework.card.Card;
import jp.co.ricoh.dsdk.scard.framework.card.CardAccessException;
import jp.co.ricoh.dsdk.scard.framework.card.CardManager;
import jp.co.ricoh.dsdk.scard.framework.card.InitializeException;
import jp.co.ricoh.dsdk.scard.framework.event.CardEvent;
import jp.co.ricoh.dsdk.scard.framework.event.CardEventListener;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceException;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceNotFoundException;
import jp.co.ricoh.dsdk.scard.framework.util.APDUException;
import jp.co.ricoh.dsdk.scard.framework.util.RequestAPDU;
import jp.co.ricoh.dsdk.scard.framework.util.ResponseAPDU;
import jp.co.ricoh.dsdk.scard.option.applet.AppletService;
import jp.co.ricoh.dsdk.xlet.UnavailableContainerException;
import jp.co.ricoh.dsdk.xlet.Xlet;
import jp.co.ricoh.dsdk.xlet.XletContext;
import jp.co.ricoh.dsdk.xlet.XletStateChangeException;

/**
 * ScfEvent
 * This is a sample application using the CardEvent of the SmartCard Framework.
 * 
 * ScfEvent
 * SmartCard Frameworkで、CardEventを利用するサンプルプログラムです。
 */
public class ScfEvent implements Xlet {
	
	/**
	 * The CommandAPDU to be sent to a JavaCard to obtain the token ID.
	 * 
	 * トークンIDを取得するためにカードに送信するコマンド
	 */
	private static final RequestAPDU requestAPDU = new RequestAPDU(new byte[]{
			(byte)0x80,(byte)0xCA,(byte)0x9F,(byte)0x7F,(byte)0x2D
	});
	
	/**
	 * The CardManager object.
	 * 
	 * CardManagerオブジェクト 
	 */
	private CardManager cardManager;
	
	/**
	 * The CardEventListener object.
	 * 
	 * CardEventListenerオブジェクト
	 */
	private CardEventListener listener;

	/**
	 * The root frame.
	 * 
	 * ルートフレーム
	 */
	private Frame frame;
	
	/**
	 * The label to be used to display a message.
	 * 
	 * メッセージ表示ラベル
	 */
	private Label msgLabel;
	
	/**
	 * The label to be used to display a token ID.
	 * 
	 * トークンID表示ラベル
	 */
	private Label tokenLabel;
	
	/**
	 * This is an initialization process.
	 * 1.obtains the root frame.
	 * 2.creates GUI.
	 * 3.creates a CardManager object.
	 * 4.sets LED state.
	 *  
	 * 初期化処理です。
	 * 1.ルートフレームを取得する。
	 * 2.GUIを作成する。
	 * 3.CardManagerオブジェクトを生成する。
	 * 4.LEDを設定する。
	 */
	synchronized public void initXlet(XletContext xletContext) throws XletStateChangeException {
		try {
			Container parent = xletContext.getContainer();
			while (!(parent instanceof Frame)) {
				parent = parent.getParent();
				if (parent == null) break;
			}
			if (parent == null) {
				return;
			}
			frame = (Frame) parent;

			createGUI();			
			
		} catch (UnavailableContainerException e) {
            throw new XletStateChangeException(e.getMessage());

        } catch (Exception e){
        	e.printStackTrace();
            throw new XletStateChangeException(e.getMessage());
        }
        
        try {        	
			cardManager = new CardManager();
				
		} catch (InitializeException e) {
			msgLabel.setText(new Text("ERR_INIT"));
			e.printStackTrace();
		}
		
		setLED(Type.START, Operation.GRN_ON);
	}

	/**
	 * Activation
	 * 
	 * 活性化。
	 */
	synchronized public void startXlet() throws XletStateChangeException {
	}
	
	/**
	 * Stop.
	 * 
	 * 停止。
	 */
	synchronized public void pauseXlet() {
	}

	/**
	 * End process.
	 * This process is similar to the process followed after the clear/stop key is pressed.
	 * 
	 * 終了処理。
	 * クリアストップキー押された時と同様の処理を行う。
	 */
	synchronized public void destroyXlet(boolean destroy) throws XletStateChangeException {
		pushClearStopKey();
	}
	
	/**
	 * GUI creation.
	 * This is called from the initXlet method, and creates GUI on the operation panel.
	 * 1.creates a title window.
	 * 2.creates a title label.
	 * 3.creates a message label.
	 * 4.creates a token ID label.
	 * 5.registers a KeyListener to the root frame for processing KeyEvents. 
	 * 
	 * GUI作成。
	 * initXletメソッドから呼び出され、オペパネにＧＵＩを作成します。
	 * 1.タイトル表示ウィンドウを作成する。
	 * 2.タイトル表示ラベルを作成する。
	 * 3.メッセージ表示ラベルを作成する。
	 * 4.トークンID表示ラベルを作成する。
	 * 5.キーリスナーを登録して、キーイベントを処理する。
	 */
	private void createGUI() {
		Window titleWindow = new Window(frame);
		titleWindow.setAttributeType(AttributeType.INVERSE);
		titleWindow.setLocation(0, 0);
		titleWindow.setSize(frame.getWidth(), 32);

		Label titleLabel = new Label(new Text("APP_TITLE"));  
		titleLabel.setFont(Font.F16);
		titleLabel.setAttributeType(AttributeType.INVERSE);
		titleLabel.setLocation(0, 0);
		titleLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(titleLabel);
		
		msgLabel = new Label(new Text("MSG_PUSHSTART"));  
		msgLabel.setFont(Font.F16);
		msgLabel.setAttributeType(AttributeType.INVERSE);
		msgLabel.setLocation(0, 16);
		msgLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(msgLabel);

		tokenLabel = new Label();
		tokenLabel.setSize(titleWindow.getWidth()-200, 16);
		tokenLabel.setLocation(
				100 ,
				(frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 2);
		frame.add(tokenLabel);
		setToken(null);
		
		frame.addKeyListener(
				new KeyAdapter() {
					public void keyPressed(KeyEvent keyEvent) {
						if (cardManager != null) {
							switch (keyEvent.getKeyCode()) {
							case KeyEvent.KEY_START:     // Start key
								pushStartKey();		
								break;
							case KeyEvent.KEY_CLR_STOP:  // Clear/Stop key
								pushClearStopKey();		
								break;
							}
						}
					}
				}
			);
	}
	
	/**
	 * The process followed after the start key is pressed.
	 * 1.registers a CardEventListener to the CardManager.
	 * 3.sets LED state.
	 * 
	 * スタートキーが、押された時の処理
	 * 1.リスナを登録する。
	 * 2.LEDを設定する。
	 */
	synchronized private void pushStartKey() {
		if (listener == null) {
			setToken(null);
			
			setMessage(new Text("MSG_SETCARD"));
			
			CardEventListener listener = new CardEventListener() {
				
				/**
				 * The process followed after a card is inserted to the card reader. 
				 * 1.obtains a Card object.
				 * 2.obtains a CardService object.
				 * 3.obtains a mutex lock of the card.
				 * 4.obtains token ID.
				 * 5.displays token ID.
				 * 6.releases the mutex lock of the card.
				 * 7.sets LED state.
				 * 
				 * カードが、認識された時の処理
				 * 1.カードを取得する。
				 * 2.カードサービスを取得する。
				 * 3.カードをロックする。
				 * 4.トークンIDを取得する。
				 * 5.トークンIDを表示する。
				 * 6.カードのロックを解除する。
				 * 7.LEDを設定する。
				 */
				public void inserted(CardEvent event) {
					Card card = event.getSlot().getCard();
					
					try {
						AppletService appletService =
							(AppletService)card.getCardService(
									AppletService.class);
						
						try {
							card.lock();
							
							ResponseAPDU responseAPDU = 
								appletService.sendRequestAPDU(requestAPDU);
							
							byte[] data = responseAPDU.data();
							
							byte[] tokenID = new byte[10];
							System.arraycopy(data, 0x03, tokenID, 0, 2);
							System.arraycopy(data, 0x05, tokenID, 2, 2);
							System.arraycopy(data, 0x13, tokenID, 4, 2);
							System.arraycopy(data, 0x0F, tokenID, 6, 4);
							
							setToken(tokenID);
							
							setMessage(new Text("MSG_SUCCESS_INSERTED"));
							
						} finally {
							card.unlock();
						}
						
					} catch (CardAccessException e) {
						setMessage(new Text("ERR_CARDACCESS"));
						e.printStackTrace();
						
					} catch (CardServiceNotFoundException e) {
						setMessage(new Text("ERR_NOCARDSERVICE"));
						e.printStackTrace();
						
					} catch (APDUException e) {
						setMessage(new Text("ERR_APDU"));
						e.printStackTrace();
						
					} catch (CardServiceException e) {
						setMessage(new Text("ERR_CARDSERVICE"));
						e.printStackTrace();
						
					}
					
					setLED(Type.START, Operation.GRN_BLINK);
				}
				
				/**
				 * The process followed after a card is removed from the card reader.
				 * 1.displays a message that indicates the removal of a card.
				 * 2.sets LED state.
				 * 
				 * カードが、外された時の処理
				 * 1.カードが外された旨を表示する。
				 * 2.LEDを設定する。
				 */
				public void removed(CardEvent event) {
					setToken(null);
					
					setMessage(new Text("MSG_SUCCESS_REMOVED"));
					
					setLED(Type.START, Operation.RED_BLINK);
				}
				
			};
			
			cardManager.addCardListener(listener);
			
			this.listener = listener;
			
			setLED(Type.START, Operation.RED_BLINK);
		}
		
	}
	
	/**
	 * The process followed after the claer/stop key is pressed.
	 * 1.removes the CardEventListener from the CardManager,
	 *   and calls abort method of the CardManager.
	 * 3.sets LED state.
	 * 
	 * クリア/ストップキーが、押された時の処理
	 * 1.リスナを削除し、abortメソッドを呼び出す。
	 * 2.LEDを設定する。
	 */
	synchronized private void pushClearStopKey() {
		if (listener != null) {
			cardManager.removeCardListener(listener);
			listener = null;
			
			cardManager.abort();
			
			setMessage(new Text("ERR_ABORT"));
			
			setLED(Type.START, Operation.GRN_ON);
		}
	}
	
	/**
	 * Displays the given token ID on the token ID label.
	 * 
	 * トークンIDを、トークンID表示ラベルに表示する。
	 */
	private void setToken(byte[] tokenID) {
		final String replace = "NUMBER";
		Text text = new Text("TOKENID_LABEL");
		
		if (tokenID == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < tokenID.length; i++) {
				n = tokenID[i];
				n &=0x000000ff;
				if(n<=0x0000000f){
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		tokenLabel.setText(text);
		tokenLabel.repaint();
	}
	
	/**
	 * Displays the given message on the message label.
	 * 
	 * メッセージを、メッセージ表示ラベルに表示する。
	 */
	private void setMessage(Text message) {
		msgLabel.setText(message);
		msgLabel.repaint();
	}
	
	/**
	 * Activates LED with the given Type and the given Operation.
	 * 
	 * LEDを動作させる。
	 */
	private void setLED(Type type, Operation ope) {
		try {
			new LED(frame, type, ope).blink();
		} catch (LEDExecutionException e) {
			e.printStackTrace();
		}
	}
}
